using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents logon request security message
	/// defined in VistALink protocol.
	/// </summary>
	public sealed class SecurityLogonRequestMessage : SecurityBaseRequestMessage
	{		
		private VistASecurityCode _accessCode, _verifyCode; 
		bool _requestVerifyCodeChange;

		/// <summary>
		/// VistALink message request type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string RequestType = "AV.Logon";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secLogonRequest.xsd";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_AVCODES_NODE_NAME = "avCodes";
		private const string XMLCONSTS_REQUEST_CVC_AVC_ATTRIBUTE_NAME = "requestCvc";
		private const string XMLCONSTS_TRUE = "true";

		/// <summary>
		/// Constructor defining minimal set of parameters required to create logon request message. 
		/// </summary>
		/// <param name="accessCode">User access code.</param>
		/// <param name="verifyCode">User verify code.</param>
		/// <param name="requestVerifyCodeChange">
		///		Boolean flag indicating that 
		///		server should prompt user to change his/her verify code when set to true.
		///	</param>
		public SecurityLogonRequestMessage( VistASecurityCode accessCode, VistASecurityCode verifyCode, bool requestVerifyCodeChange )
			: base( XsdName, RequestType )
		{	
			if( accessCode == null )
				throw( new ArgumentNullException( "accessCode" ) );

			if( verifyCode == null )
				throw( new ArgumentNullException( "verifyCode" ) );

			_accessCode = accessCode;
			_verifyCode = verifyCode;
			_requestVerifyCodeChange = requestVerifyCodeChange;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityLogonRequestMessage( XmlDocument sourceDoc )
			: base( XsdName, RequestType, sourceDoc )
		{
			XmlElement _avCodesElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( XmlParseGetRequestElement( sourceDoc ), XMLCONSTS_AVCODES_NODE_NAME);
			
			if( _requestVerifyCodeChange = _avCodesElement.HasAttribute( XMLCONSTS_REQUEST_CVC_AVC_ATTRIBUTE_NAME ) )
				XmlUtility.ParseCheckRequiredAttributeValue( _avCodesElement, XMLCONSTS_REQUEST_CVC_AVC_ATTRIBUTE_NAME, XMLCONSTS_TRUE );
			
			if( _avCodesElement.FirstChild == null || _avCodesElement.FirstChild.NodeType != XmlNodeType.CDATA )
				throw( new XmlParseException( SR.Exceptions.XmlParseCDataNotFoundWhereExpected( XMLCONSTS_AVCODES_NODE_NAME ) ) );

			try
			{			
				string[] _accessAndVerifyCodes = VistAObfuscator.Deobfuscate( _avCodesElement.FirstChild.Value ).Split( new Char[]{';'}, 3 );

				if( _accessAndVerifyCodes.Length != 2 )
					throw( new XmlParseException( SR.Exceptions.SecurityLogonRequestUnableToParseAccessCodes( _accessAndVerifyCodes.Length ) ) );

				_accessCode = VistASecurityCode.Parse( _accessAndVerifyCodes[0] );
				_verifyCode = VistASecurityCode.Parse( _accessAndVerifyCodes[1] );
			}
			catch( ArgumentException xcp )
			{
				throw( new XmlParseException( SR.Exceptions.XmlMessageParseCausedExceptionInConstructor( this.GetType().Name, xcp.Message ), xcp ) );
			}
		}

		/// <summary>
		/// XML serialization method adding access and verify codes to 
		/// request element created by the base class.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlRequestNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_AVCODES_NODE_NAME );
			
			if( _requestVerifyCodeChange )
				writer.WriteAttributeString( XMLCONSTS_REQUEST_CVC_AVC_ATTRIBUTE_NAME, XMLCONSTS_TRUE );

			// VistA access and verify codes cannot contain semicolons
			writer.WriteCData( VistAObfuscator.Obfuscate( String.Format( "{0};{1}", AccessCode.ToString(), VerifyCode.ToString() ) ) );

			writer.WriteEndElement();
		}

		/// <summary>
		/// User access code. 
		/// </summary>
		public VistASecurityCode AccessCode
		{
			get
			{
				return _accessCode;
			}
		}

		/// <summary>
		/// User verify code. 
		/// </summary>
		public VistASecurityCode VerifyCode
		{
			get
			{
				return _verifyCode;
			}
		}

		/// <summary>
		/// This value indicates if user requests 
		/// remote server to change verify code. 
		/// </summary>
		public bool RequestVerifyCodeChange
		{
			get
			{
				return _requestVerifyCodeChange;
			}
		}

	}
}
